<?php
// =============================================================================
//  CONFIGURATION
// =============================================================================

// --- 1. API Credentials ---
$VT_API_KEY = 'SET_API_KEY_HERE'; // e.g., 'a1b2c3d4...'
$VT_API_URL = 'https://www.virustotal.com/api/v3'; // e.g., Standard V3 endpoint

// --- 2. File System Paths ---
// Absolute path to store temporary files and queue data
$TEMP_UPLOAD_DIR = 'FOLDER_PATH_HERE';

// --- 3. Upload Limits (Browser -> Server) ---
$MAX_UPLOAD_SIZE_MB         = 650; // e.g., 650 (Max file size allowed)
$BROWSER_UPLOAD_CHUNK_SIZE_MB = 70;  // e.g., 70 (Chunk size for large files)
$CONCURRENT_BROWSER_UPLOADS = 2;   // e.g., 2 (Simultaneous uploads allowed)

// --- 4. VirusTotal Processing (Server -> VT) ---
$MAX_UPLOADS_PER_MINUTE    = 4;   // e.g., 4 (Public API rate limit)
$VT_DIRECT_UPLOAD_LIMIT_MB = 32;  // e.g., 32 (Files >32MB need special URL)
$UPLOAD_TIMEOUT = 600; // e.g., 600 (Timeout Server -> VT, also PHP execution time limit)

// --- 5. Queue & Retry Logic ---
$MAX_RETRIES                    = 5;   // e.g., 5 (Max attempts on API failure)
$BASE_RETRY_DELAY_SECONDS       = 60;  // e.g., 60 (Exponential backoff base)
$MAX_QUEUE_AGE_HOURS            = 3;   // e.g., 3 (Auto-delete stuck files)
$FAILED_ENTRY_RETENTION_SECONDS = 300; // e.g., 300 (Keep failed status for 5m)

// =============================================================================
//  END CONFIGURATION
// =============================================================================

// Runtime Environment
@set_time_limit($UPLOAD_TIMEOUT + 60);
@ignore_user_abort(true);

// FIX: Close session write lock to allow concurrent AJAX requests
// This prevents WordPress/PHP from forcing uploads to run one by one.
if (session_status() === PHP_SESSION_ACTIVE) {
    session_write_close();
}

$VT_API_KEY = trim($VT_API_KEY);

// Unit Conversions
$BROWSER_CHUNK_SIZE_BYTES = $BROWSER_UPLOAD_CHUNK_SIZE_MB * 1024 * 1024;
$VT_DIRECT_LIMIT_BYTES    = $VT_DIRECT_UPLOAD_LIMIT_MB * 1024 * 1024;
$MAX_UPLOAD_SIZE_BYTES    = $MAX_UPLOAD_SIZE_MB * 1024 * 1024;

// File Paths
$QUEUE_FILE      = $TEMP_UPLOAD_DIR . '/vt_upload_queue.json';
$LOCK_FILE       = $TEMP_UPLOAD_DIR . '/vt_queue.lock';
$UPLOADS_DIR     = $TEMP_UPLOAD_DIR . '/vt_uploads';
$FILES_DIR       = $TEMP_UPLOAD_DIR . '/files';
$RATE_LIMIT_FILE = $TEMP_UPLOAD_DIR . '/vt_rate_limit.json';

// Directory Setup
function ensureDirectory($dir) {
    if (!is_dir($dir)) {
        if (!mkdir($dir, 0755, true)) throw new Exception("Failed to create directory: $dir");
    }
    if (!is_writable($dir)) chmod($dir, 0755);
    if (!is_writable($dir)) throw new Exception("Directory not writable: $dir");
}

try {
    ensureDirectory($TEMP_UPLOAD_DIR);
    ensureDirectory($UPLOADS_DIR);
    ensureDirectory($FILES_DIR);
} catch (Exception $e) {
    header('Content-Type: application/json');
    die(json_encode(['success' => false, 'error' => $e->getMessage()]));
}

$endpoint = $_GET['endpoint'] ?? '';

function sanitizeUploadId($id) {
    return preg_replace('/[^a-zA-Z0-9_\.-]/', '', $id);
}

// -----------------------------------------------------------------------------
// ENDPOINT: Init Chunk Upload
// Creates a directory for the chunks and saves metadata.
// -----------------------------------------------------------------------------
if ($endpoint === 'init_chunk') {
    header('Content-Type: application/json');
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input || !isset($input['filename'], $input['filesize'], $input['total_chunks'])) {
            throw new Exception("Invalid request data");
        }
        
        $fileSize = (int)$input['filesize'];
        if ($fileSize > $MAX_UPLOAD_SIZE_BYTES) throw new Exception("File size exceeds limit");
        
        $uploadId = uniqid('chunk_', true);
        $uploadDir = $UPLOADS_DIR . '/' . $uploadId;
        
        if (!mkdir($uploadDir, 0755, true)) throw new Exception("Failed to create upload directory");
        
        $metadata = [
            'upload_id' => $uploadId,
            'filename' => basename($input['filename']),
            'filesize' => $fileSize,
            'total_chunks' => (int)$input['total_chunks'],
            'received_chunks' => 0,
            'force_upload' => $input['force_upload'] ?? false,
            'created_at' => time()
        ];
        
        file_put_contents($uploadDir . '/metadata.json', json_encode($metadata));
        echo json_encode(['success' => true, 'upload_id' => $uploadId]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Receive Chunk
// Saves a single chunk. If it's the last one, reconstructs the file.
// -----------------------------------------------------------------------------
if ($endpoint === 'upload_chunk') {
    header('Content-Type: application/json');
    try {
        $uploadId = sanitizeUploadId($_GET['upload_id'] ?? '');
        $chunkIndex = (int)($_GET['chunk_index'] ?? 0);
        
        if (empty($uploadId)) throw new Exception("Missing upload ID");
        
        $uploadDir = $UPLOADS_DIR . '/' . $uploadId;
        if (!is_dir($uploadDir)) throw new Exception("Upload directory not found");
        
        $chunkFile = $uploadDir . '/chunk_' . $chunkIndex;
        $chunkData = file_get_contents('php://input');
        
        if ($chunkData === false) throw new Exception("Failed to read chunk data");
        if (file_put_contents($chunkFile, $chunkData) === false) throw new Exception("Failed to save chunk");
        
        $metadataFile = $uploadDir . '/metadata.json';
        $metadata = json_decode(file_get_contents($metadataFile), true);
        $metadata['received_chunks']++;
        file_put_contents($metadataFile, json_encode($metadata));
        
        // Finalize if all chunks received
        if ($metadata['received_chunks'] >= $metadata['total_chunks']) {
            $finalFile = $UPLOADS_DIR . '/' . $uploadId . '_' . basename($metadata['filename']);
            $finalHandle = fopen($finalFile, 'wb');
            
            try {
                for ($i = 0; $i < $metadata['total_chunks']; $i++) {
                    $chunkPath = $uploadDir . '/chunk_' . $i;
                    if (!file_exists($chunkPath)) throw new Exception("Missing chunk $i");
                    $in = fopen($chunkPath, 'rb');
                    while (!feof($in)) fwrite($finalHandle, fread($in, 8192));
                    fclose($in);
                    unlink($chunkPath);
                }
                fclose($finalHandle);
                unlink($metadataFile);
                rmdir($uploadDir);
                chmod($finalFile, 0644);
                
                $queueId = addToQueue($metadata['filename'], $finalFile, $metadata['filesize'], $metadata['force_upload'] ?? false);
                echo json_encode(['success' => true, 'upload_id' => $queueId, 'message' => 'Uploaded and queued']);
            } catch (Exception $e) {
                if (is_resource($finalHandle)) fclose($finalHandle);
                if (file_exists($finalFile)) unlink($finalFile);
                throw $e;
            }
        } else {
            echo json_encode(['success' => true, 'message' => 'Chunk received']);
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Standard Upload
// Handles non-chunked, standard POST file uploads.
// -----------------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_GET['endpoint'])) {
    header('Content-Type: application/json');
    $tempFilePath = null;
    try {
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            throw new Exception("File upload failed code: " . ($_FILES['file']['error'] ?? 'Unknown'));
        }
        $tempFilePath = $_FILES['file']['tmp_name'];
        if ($_FILES['file']['size'] > $MAX_UPLOAD_SIZE_BYTES) throw new Exception("File too large");
        
        $forceUpload = isset($_POST['force_upload']) && $_POST['force_upload'] === '1';
        $uploadId = addToQueue($_FILES['file']['name'], $tempFilePath, $_FILES['file']['size'], $forceUpload);
        
        echo json_encode(['success' => true, 'upload_id' => $uploadId, 'message' => 'File queued']);
    } catch (Exception $e) {
        if ($tempFilePath && file_exists($tempFilePath)) @unlink($tempFilePath);
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Check Status
// Returns the status of a specific file in the queue.
// -----------------------------------------------------------------------------
if (isset($_GET['endpoint']) && $_GET['endpoint'] === 'status' && isset($_GET['upload_id'])) {
    header('Content-Type: application/json');
    $status = getUploadStatus($_GET['upload_id']);
    if ($status) {
        echo json_encode(['success' => true, 'status' => $status]);
    } else {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Upload not found']);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Process Queue (Worker)
// Called periodically via AJAX to process pending files in the queue.
// -----------------------------------------------------------------------------
if (isset($_GET['endpoint']) && $_GET['endpoint'] === 'process_queue') {
    header('Content-Type: application/json');
    $lockFp = fopen($LOCK_FILE, 'w+');
    
    // Non-blocking lock to ensure only one worker runs at a time
    if (!flock($lockFp, LOCK_EX | LOCK_NB)) {
        fclose($lockFp);
        echo json_encode(['success' => true, 'worker_busy' => true, 'rate_limit_info' => getRateLimitInfo(), 'counts' => getQueueCounts()]);
        exit;
    }
    try {
        $queue = processQueue();
        echo json_encode([
            'success' => true, 'worker_busy' => false, 'queue' => array_slice($queue, -10),
            'counts' => getQueueCounts(), 'rate_limit_info' => getRateLimitInfo()
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    } finally {
        flock($lockFp, LOCK_UN);
        fclose($lockFp);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Acknowledge Completion
// Removes the file from the queue and server after UI confirms receipt.
// -----------------------------------------------------------------------------
if (isset($_GET['endpoint']) && $_GET['endpoint'] === 'ack_completion' && isset($_GET['upload_id'])) {
    header('Content-Type: application/json');
    try {
        $uploadId = $_GET['upload_id'];
        updateQueue(function($queue) use ($uploadId) {
            foreach ($queue as $key => $entry) {
                if ($entry['id'] === $uploadId) {
                    if (isset($entry['file_path']) && file_exists($entry['file_path'])) @unlink($entry['file_path']);
                    unset($queue[$key]);
                }
            }
            return array_values($queue);
        });
        echo json_encode(['success' => true]);
    } catch (Exception $e) { echo json_encode(['success' => false, 'error' => $e->getMessage()]); }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Purge Queue
// Manually clears all items and deletes associated files.
// -----------------------------------------------------------------------------
if (isset($_GET['endpoint']) && $_GET['endpoint'] === 'purge_queue') {
    header('Content-Type: application/json');
    try { echo json_encode(purgeQueue()); } 
    catch (Exception $e) { http_response_code(500); echo json_encode(['success' => false, 'error' => $e->getMessage()]); }
    exit;
}

// =============================================================================
// HELPER FUNCTIONS
// =============================================================================

// Reads the rate limit file to check API usage
function getRateLimitInfo() {
    global $RATE_LIMIT_FILE, $MAX_UPLOADS_PER_MINUTE;
    $fp = fopen($RATE_LIMIT_FILE, 'c+'); flock($fp, LOCK_SH); $content = stream_get_contents($fp); flock($fp, LOCK_UN); fclose($fp);
    $data = $content ? json_decode($content, true) : null; $currentTime = time();
    if (!$data || $currentTime >= $data['reset_time']) return ['uploads_this_minute' => 0, 'reset_time' => $currentTime + 60, 'max_uploads_per_minute' => $MAX_UPLOADS_PER_MINUTE, 'can_upload' => true];
    return ['uploads_this_minute' => $data['uploads_this_minute'], 'reset_time' => $data['reset_time'], 'max_uploads_per_minute' => $MAX_UPLOADS_PER_MINUTE, 'can_upload' => $data['uploads_this_minute'] < $MAX_UPLOADS_PER_MINUTE];
}

// Updates the rate limit counter safely
function incrementUploadCount() {
    global $RATE_LIMIT_FILE; $fp = fopen($RATE_LIMIT_FILE, 'c+');
    if (flock($fp, LOCK_EX)) {
        $content = stream_get_contents($fp); $data = $content ? json_decode($content, true) : null; $currentTime = time();
        if (!$data || $currentTime >= $data['reset_time']) $data = ['uploads_this_minute' => 1, 'reset_time' => $currentTime + 60];
        else $data['uploads_this_minute']++;
        ftruncate($fp, 0); rewind($fp); fwrite($fp, json_encode($data)); flock($fp, LOCK_UN);
    }
    fclose($fp);
}

// Moves uploaded file to permanent storage and adds to JSON queue
function addToQueue($originalFileName, $tempFilePath, $fileSize, $forceUpload = false) {
    global $FILES_DIR; $uploadId = uniqid('vt_', true);
    $permanentPath = $FILES_DIR . '/' . $uploadId . '_' . basename($originalFileName);
    if (!rename($tempFilePath, $permanentPath)) throw new Exception("Failed to move file");
    chmod($permanentPath, 0644);
    $entry = [
        'id' => $uploadId, 
        'file_path' => $permanentPath, 
        'original_name' => $originalFileName, 
        'file_size' => $fileSize, 
        'status' => 'queued', 
        'detailed_status' => 'In Queue', 
        'force_upload' => $forceUpload,
        'created_at' => time(), 
        'attempts' => 0, 
        'last_attempt' => null, 
        'next_attempt' => null
    ];
    updateQueue(function($queue) use ($entry) { $queue[] = $entry; return $queue; });
    return $uploadId;
}

// Safely updates the queue file with a callback function
function updateQueue($callback) {
    global $QUEUE_FILE; $fp = fopen($QUEUE_FILE, 'c+');
    if (flock($fp, LOCK_EX)) {
        $content = stream_get_contents($fp); $queue = $content ? json_decode($content, true) : [];
        if (!is_array($queue)) $queue = [];
        $queue = $callback($queue);
        ftruncate($fp, 0); rewind($fp); fwrite($fp, json_encode($queue, JSON_PRETTY_PRINT)); flock($fp, LOCK_UN);
    }
    fclose($fp); return $queue;
}

// Reads queue without locking
function loadQueue() {
    global $QUEUE_FILE; if (!file_exists($QUEUE_FILE)) return [];
    $fp = fopen($QUEUE_FILE, 'r'); flock($fp, LOCK_SH); $content = stream_get_contents($fp); flock($fp, LOCK_UN); fclose($fp);
    return json_decode($content, true) ?: [];
}

// Counts active items in queue
function getQueueCounts() {
    $q = loadQueue(); $active = 0;
    foreach($q as $e) if(in_array($e['status'], ['queued','retrying','processing'])) $active++;
    return ['active' => $active];
}

// Finds a specific upload by ID
function getUploadStatus($uploadId) {
    $queue = loadQueue(); foreach ($queue as $entry) if ($entry['id'] === $uploadId) return $entry;
    return null;
}

// Deletes all files and clears queue
function purgeQueue() {
    global $QUEUE_FILE; $deleted = 0;
    updateQueue(function($queue) use (&$deleted) {
        foreach ($queue as $entry) if (isset($entry['file_path']) && file_exists($entry['file_path'])) { @unlink($entry['file_path']); $deleted++; }
        return [];
    });
    return ['message' => "Purged $deleted files.", 'files_deleted' => $deleted];
}

// Main logic: Iterates queue and talks to VT API
function processQueue() {
    global $VT_API_KEY, $VT_API_URL, $UPLOAD_TIMEOUT, $MAX_RETRIES, $BASE_RETRY_DELAY_SECONDS, 
           $MAX_QUEUE_AGE_HOURS, $MAX_UPLOADS_PER_MINUTE, $FAILED_ENTRY_RETENTION_SECONDS, $VT_DIRECT_LIMIT_BYTES;
    $startTime = time(); $timeLimit = 45; 
    
    updateQueue(function($queue) use (
        $VT_API_KEY, $VT_API_URL, $UPLOAD_TIMEOUT, $MAX_RETRIES, $BASE_RETRY_DELAY_SECONDS, 
        $MAX_QUEUE_AGE_HOURS, $MAX_UPLOADS_PER_MINUTE, $FAILED_ENTRY_RETENTION_SECONDS, $VT_DIRECT_LIMIT_BYTES, $startTime, $timeLimit
    ) {
        $currentTime = time(); $rateInfo = getRateLimitInfo(); $processed = 0;
        $allowed = min($rateInfo['max_uploads_per_minute'] - $rateInfo['uploads_this_minute'], $MAX_UPLOADS_PER_MINUTE);

        foreach ($queue as $key => &$entry) {
            // Time constraints
            if ((time() - $startTime) > $timeLimit) break;
            
            // Cleanup Logic
            if (in_array($entry['status'], ['completed', 'failed'])) {
                if (($currentTime - ($entry['completed_at'] ?? $entry['created_at'])) > $FAILED_ENTRY_RETENTION_SECONDS) {
                    if (isset($entry['file_path']) && file_exists($entry['file_path'])) @unlink($entry['file_path']);
                    unset($queue[$key]); continue;
                }
            }
            if (($currentTime - $entry['created_at']) > ($MAX_QUEUE_AGE_HOURS * 3600)) {
                if (isset($entry['file_path']) && file_exists($entry['file_path'])) @unlink($entry['file_path']);
                unset($queue[$key]); continue;
            }
            
            if (in_array($entry['status'], ['completed', 'failed'])) continue;
            if ($processed >= $allowed) break;

            // Processing Logic
            if ($entry['status'] === 'queued' || ($entry['status'] === 'retrying' && $currentTime >= $entry['next_attempt'])) {
                $entry['attempts']++; $entry['last_attempt'] = $currentTime; $entry['status'] = 'processing';
                
                $forceUpload = isset($entry['force_upload']) && $entry['force_upload'];
                $existingReport = null;

                try {
                    if (!file_exists($entry['file_path'])) throw new Exception("File not found on server");
                    $uploader = new VirusTotalUploader($VT_API_KEY, $VT_API_URL, $UPLOAD_TIMEOUT, $VT_DIRECT_LIMIT_BYTES);

                    // 1. Check Hash (if not forced)
                    if (!$forceUpload) {
                        $entry['detailed_status'] = 'Checking hash...'; 
                        $fileHash = hash_file('sha256', $entry['file_path']);
                        $existingReport = $uploader->getFileReport($fileHash);
                    }

                    if ($existingReport) {
                          // File exists on VT
                          $entry['status'] = 'completed'; 
                          $entry['detailed_status'] = 'Found existing report';
                          $entry['result'] = $existingReport; 
                          $entry['completed_at'] = $currentTime;
                          if (file_exists($entry['file_path'])) @unlink($entry['file_path']);
                    } else {
                          // 2. Upload File
                          $entry['detailed_status'] = $forceUpload ? 'Forcing new upload...' : 'Sending to VirusTotal...';
                          $processed++; 
                          $result = $uploader->uploadFile($entry['file_path'], $entry['original_name']);
                          $entry['status'] = 'completed'; $entry['detailed_status'] = 'Upload Complete';
                          $entry['result'] = $result; $entry['completed_at'] = $currentTime; incrementUploadCount();
                          if (file_exists($entry['file_path'])) @unlink($entry['file_path']);
                    }

                } catch (Exception $e) {
                    // Error Handling
                    $entry['error'] = $e->getMessage(); $entry['status'] = 'failed'; $entry['detailed_status'] = 'Failed: ' . substr($e->getMessage(), 0, 30) . '...';
                    $transientErrors = ['rate limit', 'cURL error', 'timeout', '429', '502', '503', 'deadline'];
                    foreach ($transientErrors as $err) {
                        if (stripos($e->getMessage(), $err) !== false && $entry['attempts'] < $MAX_RETRIES) {
                            $entry['status'] = 'retrying'; $entry['detailed_status'] = 'Retrying (Error: ' . $err . ')';
                            $entry['next_attempt'] = $currentTime + ($BASE_RETRY_DELAY_SECONDS * pow(2, $entry['attempts'] - 1));
                            break;
                        }
                    }
                    if ($entry['status'] === 'failed') {
                        $entry['completed_at'] = $currentTime;
                        if (file_exists($entry['file_path'])) @unlink($entry['file_path']);
                    }
                }
            }
        }
        return array_values($queue);
    });
    return loadQueue();
}

class VirusTotalUploader {
    private $apiKey; private $apiUrl; private $uploadTimeout; private $limitBytes;
    public function __construct($apiKey, $apiUrl, $uploadTimeout, $limitBytes) {
        $this->apiKey = $apiKey; $this->apiUrl = $apiUrl; $this->uploadTimeout = $uploadTimeout; $this->limitBytes = $limitBytes;
    }
    
    // Checks if file already exists on VT
    public function getFileReport($hash) {
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $this->apiUrl . '/files/' . $hash,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => ['x-apikey: ' . $this->apiKey, 'Accept: application/json'],
            CURLOPT_TIMEOUT => 15
        ]);
        $response = curl_exec($curl); $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE); curl_close($curl);
        
        if ($httpCode === 404) return false; 
        return $this->handleResponse($response, $httpCode, "Check Hash");
    }

    // Router for upload method based on size
    public function uploadFile($filePath, $originalFileName = null) {
        return (filesize($filePath) > $this->limitBytes) ? $this->largeFileUpload($filePath, $originalFileName) : $this->simpleUpload($filePath, $originalFileName);
    }
    
    // Standard upload (<32MB)
    private function simpleUpload($filePath, $originalFileName = null) {
        $curl = curl_init(); $postname = $originalFileName ? $originalFileName : basename($filePath);
        $cFile = new CURLFile($filePath, 'application/octet-stream', $postname);
        curl_setopt_array($curl, [CURLOPT_URL => $this->apiUrl . '/files', CURLOPT_POST => true, CURLOPT_POSTFIELDS => ['file' => $cFile], CURLOPT_HTTPHEADER => ['x-apikey: ' . $this->apiKey, 'Accept: application/json'], CURLOPT_RETURNTRANSFER => true, CURLOPT_TIMEOUT => $this->uploadTimeout]);
        $response = curl_exec($curl); $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE); $error = curl_error($curl); curl_close($curl);
        if ($response === false) throw new Exception("cURL error: " . $error);
        if ($httpCode === 413) return $this->largeFileUpload($filePath, $originalFileName);
        return $this->handleResponse($response, $httpCode);
    }
    
    // Large file upload (>32MB)
    private function largeFileUpload($filePath, $originalFileName = null) {
        // Step 1: Get upload URL
        $curl = curl_init();
        curl_setopt_array($curl, [CURLOPT_URL => $this->apiUrl . '/files/upload_url', CURLOPT_RETURNTRANSFER => true, CURLOPT_HTTPHEADER => ['x-apikey: ' . $this->apiKey, 'Accept: application/json'], CURLOPT_TIMEOUT => 30]);
        $response = curl_exec($curl); $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE); curl_close($curl);
        $data = $this->handleResponse($response, $httpCode, "Get URL"); $uploadUrl = $data['data'];
        
        // Step 2: Upload to that URL
        $curl = curl_init(); $postname = $originalFileName ? $originalFileName : basename($filePath);
        $cFile = new CURLFile($filePath, 'application/octet-stream', $postname);
        curl_setopt_array($curl, [CURLOPT_URL => $uploadUrl, CURLOPT_POST => true, CURLOPT_POSTFIELDS => ['file' => $cFile], CURLOPT_HTTPHEADER => ['x-apikey: ' . $this->apiKey, 'Accept: application/json'], CURLOPT_RETURNTRANSFER => true, CURLOPT_TIMEOUT => $this->uploadTimeout]);
        $response = curl_exec($curl); $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE); $error = curl_error($curl); curl_close($curl);
        if ($response === false) throw new Exception("Upload failed: " . $error);
        return $this->handleResponse($response, $httpCode, "Upload Phase");
    }
    
    private function handleResponse($response, $httpCode, $context = "") {
        $contextMsg = $context ? " [$context]" : "";
        if ($httpCode === 401) throw new Exception("Invalid API Key (401)$contextMsg");
        if ($httpCode === 429) throw new Exception("API rate limit exceeded (429)$contextMsg");
        $result = json_decode($response, true);
        if ($httpCode < 200 || $httpCode >= 300) throw new Exception(($result['error']['message'] ?? "HTTP Error $httpCode") . $contextMsg);
        if (!$result) throw new Exception("Invalid JSON response$contextMsg");
        return $result;
    }
}

if (rand(1, 50) === 1) cleanupOldFiles();
function cleanupOldFiles() {
    global $UPLOADS_DIR;
    foreach (glob($UPLOADS_DIR . '/chunk_*', GLOB_ONLYDIR) as $dir) {
        if (time() - filemtime($dir) > 7200) { array_map('unlink', glob("$dir/*")); rmdir($dir); }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>VirusTotal File Uploader</title>
    <link rel="icon" href="data:image/svg+xml,%3Csvg width='1em' height='1em' xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 89'%3E%3Cpath fill='%230b4dda' fill-rule='evenodd' d='M45.292 44.5 0 89h100V0H0l45.292 44.5zM90 80H22l35.987-35.2L22 9h68v71z'/%3E%3C/svg%3E">
    <style>
        :root {
            /* Light Mode - WCAG AA/AAA Compliant Colors */
            --bg-color: #f0f2f5; 
            --text-color: #212529; 
            --card-bg: #ffffff;
            --card-shadow: 0 2px 10px rgba(0,0,0,0.1); 
            --border-color: #dee2e6;
            --primary-color: #007cba; 
            --primary-hover: #005a87;
            --accent-bg: #e9ecef; 
            
            /* Status Colors (Light) */
            --success-bg: #d1e7dd; 
            --success-text: #0a3622; /* High contrast green */
            
            --failed-bg: #f8d7da; 
            --failed-text: #58151c; /* High contrast red */
            
            --processing-bg: #fff3cd; 
            --processing-text: #664d03; /* High contrast yellow/brown */
            
            --drop-hover-bg: #f0f8ff;
        }
        
        @media (prefers-color-scheme: dark) {
            :root {
                /* Dark Mode - High Contrast */
                --bg-color: #121212; 
                --text-color: #e0e0e0; 
                --card-bg: #1e1e1e;
                --card-shadow: 0 2px 10px rgba(0,0,0,0.5); 
                --border-color: #444;
                --primary-color: #4da3ff; 
                --primary-hover: #2b80d6;
                --accent-bg: #2d2d2d; 
                
                /* Status Colors (Dark - Deep backgrounds with pastel text for readability) */
                --success-bg: #051b11; 
                --success-text: #75b798; 
                
                --failed-bg: #2c0b0e; 
                --failed-text: #ea868f; 
                
                --processing-bg: #332701; 
                --processing-text: #ffda6a;
                
                --drop-hover-bg: #252525;
            }
        }
        
        body { 
            font-family: Inter, Roboto, "Helvetica Neue", Arial, sans-serif;
            max-width: 1000px; margin: 0 auto; padding: 20px; 
            background-color: var(--bg-color); color: var(--text-color); 
            transition: 0.3s; 
        }
        h1 { margin-top: 0; margin-bottom: 20px; }
        .upload-container { 
            background: var(--card-bg); border-radius: 8px; padding: 30px; 
            box-shadow: var(--card-shadow); text-align: center; 
        }
        .options-area {
            text-align: left; margin-bottom: 10px; padding: 10px;
            background: var(--accent-bg); border-radius: 4px;
            display: flex; align-items: center; gap: 8px; font-size: 14px;
        }
        .drop-area { 
            border: 3px dashed var(--border-color); border-radius: 8px; 
            padding: 40px; cursor: pointer; transition: 0.3s; margin-bottom: 20px; 
        }
        .drop-area:hover, .drop-area.dragover, .drop-area:focus-visible { 
            border-color: var(--primary-color); background-color: var(--drop-hover-bg); 
            outline: none; 
        }
        .upload-btn { 
            background-color: var(--primary-color); color: white; border: none; 
            padding: 12px 24px; border-radius: 4px; cursor: pointer; margin: 10px 0; 
            pointer-events: none; /* Let clicks pass to the container */
        }
        .info-container { display: flex; gap: 20px; margin: 20px 0; }
        .info-column { 
            flex: 1; padding: 10px; background-color: var(--accent-bg); 
            border-radius: 4px; text-align: center; 
        }
        .info-column h3 { margin: 0 0 5px 0; font-size: 14px; }
        .info-column p { margin: 2px 0; font-size: 13px; }
        
        /* Accessible File List */
        .file-list { 
            text-align: left; margin-top: 20px; 
            list-style: none; padding: 0; 
        }
        .file-item { 
            padding: 10px; border: 1px solid var(--border-color); 
            border-radius: 4px; margin-bottom: 8px; background-color: var(--card-bg); 
            display: flex; flex-direction: column; gap: 5px;
        }
        .file-header { display: flex; justify-content: space-between; align-items: center; }
        .file-name { font-family: monospace; font-weight: bold; }
        .file-status { font-size: 14px; display: flex; align-items: center; gap: 6px; }
        
        .file-item.completed { background-color: var(--success-bg); color: var(--success-text); }
        /* Hide progress bar when completed */
        .file-item.completed progress { display: none; }
        
        .file-item.failed { background-color: var(--failed-bg); color: var(--failed-text); }
        .file-item.processing { background-color: var(--processing-bg); color: var(--processing-text); }
        
        progress { width: 100%; height: 8px; border-radius: 4px; margin-top: 5px; }
        progress::-webkit-progress-bar { background-color: var(--accent-bg); border-radius: 4px; }
        progress::-webkit-progress-value { background-color: var(--primary-color); border-radius: 4px; }
        
        .purge-btn { 
            background-color: #dc3545; color: white; border: none; 
            padding: 8px 16px; border-radius: 4px; cursor: pointer; width: 100%; 
            font-size: 13px;
        }
        .purge-btn:hover { background-color: #bb2d3b; }
        *:focus-visible { outline: 2px solid var(--primary-color); outline-offset: 2px; }
        a { color: inherit; text-decoration: underline; font-weight: bold; }
        a:hover { text-decoration: none; }
    </style>
</head>
<body>
    <main class="upload-container">
        <header>
            <h1>VirusTotal File Uploader</h1>
        </header>
        
        <div class="options-area">
            <label>
                <input type="checkbox" id="forceUpload"> Force Re-scan (Ignore existing results)
            </label>
        </div>

        <div class="drop-area" id="dropArea" tabindex="0" role="button" aria-label="Drop files here or click to upload">
            <p>Drag & drop files here</p>
            <div class="upload-btn">Browse Files</div>
            <input type="file" id="fileInput" style="display:none" multiple>
        </div>
        
        <section class="info-container" aria-label="System Status">
            <div class="info-column">
                <h3>Status</h3>
                <p id="queueStatusText" aria-live="polite">Idle</p>
                <p>Active: <span id="totalCount">0</span></p>
            </div>
            <div class="info-column">
                <h3>Rate Limit</h3>
                <p>Used: <span id="uploadsCount">0</span>/<span id="maxUploads">4</span></p>
                <p>Reset: <span id="resetTime">-</span></p>
            </div>
            <div class="info-column">
                <h3>Action</h3>
                <button class="purge-btn" id="purgeBtn">Purge Queue</button>
            </div>
        </section>
        
        <ul class="file-list" id="fileListContainer" aria-live="polite" aria-relevant="additions">
        </ul>
    </main>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const dropArea = document.getElementById('dropArea');
            const fileInput = document.getElementById('fileInput');
            const fileListContainer = document.getElementById('fileListContainer');
            const forceUploadCheckbox = document.getElementById('forceUpload');
            
            // Configuration injected from PHP
            const MAX_CONCURRENT_UPLOADS = <?php echo $CONCURRENT_BROWSER_UPLOADS; ?>;
            
            let pendingQueue = []; 
            let activeUploads = 0; 

            // Click handling for drop area
            dropArea.addEventListener('click', () => fileInput.click());
            
            // Keyboard access for drop area
            dropArea.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    fileInput.click();
                }
            });

            fileInput.addEventListener('change', handleFileSelect);
            document.getElementById('purgeBtn').addEventListener('click', purgeQueue);

            // Drag & Drop visuals
            ['dragenter', 'dragover'].forEach(evt => {
                dropArea.addEventListener(evt, (e) => { 
                    e.preventDefault(); 
                    dropArea.classList.add('dragover'); 
                });
            });
            ['dragleave', 'drop'].forEach(evt => {
                dropArea.addEventListener(evt, (e) => { 
                    e.preventDefault(); 
                    dropArea.classList.remove('dragover'); 
                });
            });

            dropArea.addEventListener('drop', (e) => {
                if(e.dataTransfer.files.length) handleFiles(e.dataTransfer.files);
            });

            function handleFileSelect(e) { if (e.target.files.length) handleFiles(e.target.files); }
            
            function handleFiles(files) {
                const forceState = forceUploadCheckbox.checked;
                Array.from(files).forEach(file => {
                    const id = 'ui-' + Math.random().toString(36).substr(2, 9);
                    addToFileList(id, file.name);
                    queueUpload(file, id, forceState);
                });
            }

            function queueUpload(file, uiId, forceState) {
                pendingQueue.push({file: file, id: uiId, force: forceState});
                updateStatus(uiId, "Waiting in queue...", 0);
                processQueue();
            }

            function processQueue() {
                if (activeUploads >= MAX_CONCURRENT_UPLOADS) return;
                if (pendingQueue.length === 0) return;

                const task = pendingQueue.shift();
                activeUploads++;
                
                performUpload(task.file, task.id, task.force)
                    .finally(() => {
                        activeUploads--;
                        processQueue();
                    });
            }

            function performUpload(file, uiId, forceState) {
                return new Promise((resolve) => {
                    const onComplete = () => resolve();
                    const browserLimit = <?php echo $BROWSER_CHUNK_SIZE_BYTES; ?>;
                    
                    if (file.size > browserLimit) {
                        uploadChunked(file, uiId, forceState, onComplete);
                    } else {
                        uploadStandard(file, uiId, forceState, onComplete);
                    }
                });
            }

            function addToFileList(id, name) {
                const el = document.createElement('li');
                el.className = 'file-item'; el.id = id;
                el.innerHTML = `
                    <div class="file-header">
                        <span class="file-name">${name}</span>
                    </div>
                    <div class="file-status"><span>Preparing...</span></div>
                    <progress max="100" value="0" aria-label="Upload progress"></progress>
                `;
                fileListContainer.prepend(el);
            }

            function updateStatus(id, msg, percent = null, type = '') {
                const el = document.getElementById(id);
                if (el) {
                    let icon = '';
                    if (type === 'completed') icon = '✓ ';
                    else if (type === 'failed') icon = '✕ ';
                    else if (type === 'processing') icon = '⟳ ';

                    el.querySelector('.file-status span').innerHTML = icon + msg;
                    if(type) el.className = `file-item ${type}`;
                    
                    if (percent !== null) {
                        const progressBar = el.querySelector('progress');
                        if(progressBar) {
                            if (percent === -1) progressBar.removeAttribute('value'); // Indeterminate state
                            else progressBar.value = percent;
                        }
                    }
                }
            }

            function uploadStandard(file, uiId, forceState, callback) {
                const formData = new FormData();
                formData.append('file', file);
                formData.append('force_upload', forceState ? '1' : '0');

                const xhr = new XMLHttpRequest();
                xhr.upload.addEventListener('progress', (e) => {
                    if(e.lengthComputable) {
                        const pct = Math.round((e.loaded/e.total)*100);
                        updateStatus(uiId, `Uploading: ${pct}%`, pct);
                    }
                });
                xhr.onload = () => {
                    try {
                        const r = JSON.parse(xhr.responseText);
                        if(r.success) { 
                            // Start polling immediately after getting upload ID
                            pollStatus(r.upload_id, uiId); 
                            updateStatus(uiId, 'Queued for VirusTotal...', -1);
                        } else {
                            updateStatus(uiId, 'Error: '+r.error, 0, 'failed');
                        }
                    } catch(e) { updateStatus(uiId, 'Server Error', 0, 'failed'); }
                    callback();
                };
                xhr.onerror = () => { updateStatus(uiId, 'Network Error', 0, 'failed'); callback(); };
                xhr.open('POST', '');
                xhr.send(formData);
            }

            function uploadChunked(file, uiId, forceState, callback) {
                const chunkSize = <?php echo $BROWSER_CHUNK_SIZE_BYTES; ?>;
                const totalChunks = Math.ceil(file.size / chunkSize);
                
                fetch('?endpoint=init_chunk', {
                    method: 'POST',
                    body: JSON.stringify({
                        filename: file.name, 
                        filesize: file.size, 
                        total_chunks: totalChunks,
                        force_upload: forceState
                    })
                }).then(r=>r.json()).then(d => {
                    if(!d.success) throw new Error(d.error);
                    // FIXED: Do not poll here. invalid ID at this stage.
                    sendChunk(d.upload_id, 0);
                }).catch(e => {
                    updateStatus(uiId, 'Init Fail: '+e.message, 0, 'failed');
                    callback();
                });

                function sendChunk(uid, idx) {
                    const start = idx * chunkSize;
                    const chunk = file.slice(start, Math.min(start + chunkSize, file.size));
                    
                    const xhr = new XMLHttpRequest();
                    xhr.open('POST', `?endpoint=upload_chunk&upload_id=${uid}&chunk_index=${idx}`);
                    
                    xhr.upload.onprogress = (e) => {
                        if (e.lengthComputable) {
                            const totalLoaded = start + e.loaded;
                            const percent = Math.min(100, Math.round((totalLoaded / file.size) * 100));
                            updateStatus(uiId, `Uploading: ${percent}%`, percent);
                        }
                    };

                    xhr.onload = () => {
                        try {
                            const d = JSON.parse(xhr.responseText);
                            if(!d.success) throw new Error(d.error);
                            
                            if(d.upload_id && d.message.includes('queued')) {
                                updateStatus(uiId, 'Queued for VirusTotal...', -1);
                                pollStatus(d.upload_id, uiId);
                                callback();
                            } else {
                                sendChunk(uid, idx+1);
                            }
                        } catch (e) {
                            updateStatus(uiId, 'Chunk Fail: '+e.message, 0, 'failed');
                            callback();
                        }
                    };
                    
                    xhr.onerror = () => {
                        updateStatus(uiId, 'Network Error during upload', 0, 'failed');
                        callback();
                    };

                    xhr.send(chunk);
                }
            }

            function pollStatus(sid, uid) {
                const interval = setInterval(() => {
                    fetch(`?endpoint=status&upload_id=${sid}`).then(r=>r.json()).then(d => {
                        if(d.success && d.status) {
                            const s = d.status;
                            if(s.status === 'completed') {
                                clearInterval(interval);
                                const res = s.result.data;
                                let url = '';
                                if (res.type === 'file') {
                                    url = `https://www.virustotal.com/gui/file/${res.id}`;
                                } else {
                                    url = `https://www.virustotal.com/gui/file-analysis/${res.id}`;
                                }
                                const statusMsg = s.detailed_status === 'Found existing report' ? 'Already Scanned.' : 'Done.';
                                updateStatus(uid, `${statusMsg} <a href="${url}" target="_blank">View Analysis »</a>`, 100, 'completed');
                                fetch(`?endpoint=ack_completion&upload_id=${sid}`);
                            } else if(s.status === 'failed') {
                                clearInterval(interval);
                                updateStatus(uid, 'Failed: '+s.error, 0, 'failed');
                                fetch(`?endpoint=ack_completion&upload_id=${sid}`);
                            } else {
                                const msg = s.detailed_status || `Processing (Attempt ${s.attempts})`;
                                updateStatus(uid, msg, -1, 'processing');
                            }
                        } else { clearInterval(interval); }
                    }).catch(() => clearInterval(interval));
                }, 4000);
            }

            function purgeQueue() {
                if(confirm('Clear queue history? Active uploads will stop.')) {
                    fetch('?endpoint=purge_queue').then(r=>r.json()).then(d => {
                        fileListContainer.innerHTML = '';
                        pendingQueue = []; 
                        activeUploads = 0;
                        alert(d.message);
                    });
                }
            }

            setInterval(() => {
                fetch('?endpoint=process_queue').then(r=>r.json()).then(d => {
                    if(d.success) {
                        document.getElementById('totalCount').innerText = d.counts ? d.counts.active : 0;
                        document.getElementById('uploadsCount').innerText = d.rate_limit_info.uploads_this_minute;
                        
                        const statusText = d.worker_busy ? 'Processing Queue...' : 
                                         (d.counts && d.counts.active > 0) ? 'Waiting for worker...' : 'Idle';
                        document.getElementById('queueStatusText').innerText = statusText;
                        
                        if(d.rate_limit_info.reset_time) {
                            document.getElementById('resetTime').innerText = new Date(d.rate_limit_info.reset_time * 1000).toLocaleTimeString();
                        }
                    }
                });
            }, 5000);
        });
    </script>
</body>
</html>