<?php
// =============================================================================
// 1. API CREDENTIALS
// =============================================================================
$VT_API_KEY = 'APIKEYHERE'; # Get it from your user account
$VT_API_URL = 'https://www.virustotal.com/api/v3';

// =============================================================================
// 2. UPLOAD BEHAVIOR SETTINGS
// =============================================================================
// [Browser -> Server] Keep low to avoid timeouts
$BROWSER_UPLOAD_CHUNK_SIZE_MB = 70; # Keep lower than your server's limit

// [Server -> VirusTotal] VT requires large file API > 32MB
$VT_DIRECT_UPLOAD_LIMIT_MB = 32; 

// [Timeouts]
$UPLOAD_TIMEOUT = 600;

// =============================================================================
// 3. SERVER & SYSTEM SETTINGS
// =============================================================================
$MAX_UPLOAD_SIZE_MB = 650; 
$TEMP_UPLOAD_DIR = 'FOLDERPATHHERE'; # temporary folder to store files, e.g.: '/srv/username/temp'

// =============================================================================
// 4. QUEUE & RETRY LOGIC
// =============================================================================
$MAX_RETRIES = 5;
$BASE_RETRY_DELAY_SECONDS = 60;
$MAX_UPLOADS_PER_MINUTE = 4;
$MAX_QUEUE_AGE_HOURS = 3;
$FAILED_ENTRY_RETENTION_SECONDS = 300; 

// =============================================================================
// END CONFIGURATION
// =============================================================================

// Safety & Limits
@set_time_limit($UPLOAD_TIMEOUT + 60);
@ignore_user_abort(true);

$VT_API_KEY = trim($VT_API_KEY);

// Byte Calculations
$BROWSER_CHUNK_SIZE_BYTES = $BROWSER_UPLOAD_CHUNK_SIZE_MB * 1024 * 1024;
$VT_DIRECT_LIMIT_BYTES    = $VT_DIRECT_UPLOAD_LIMIT_MB * 1024 * 1024;
$MAX_UPLOAD_SIZE_BYTES    = $MAX_UPLOAD_SIZE_MB * 1024 * 1024;

// Paths
$QUEUE_FILE      = $TEMP_UPLOAD_DIR . '/vt_upload_queue.json';
$LOCK_FILE       = $TEMP_UPLOAD_DIR . '/vt_queue.lock';
$UPLOADS_DIR     = $TEMP_UPLOAD_DIR . '/vt_uploads';
$FILES_DIR       = $TEMP_UPLOAD_DIR . '/files';
$RATE_LIMIT_FILE = $TEMP_UPLOAD_DIR . '/vt_rate_limit.json';

// Init Directories
function ensureDirectory($dir) {
    if (!is_dir($dir)) {
        if (!mkdir($dir, 0755, true)) throw new Exception("Failed to create directory: $dir");
    }
    if (!is_writable($dir)) chmod($dir, 0755);
    if (!is_writable($dir)) throw new Exception("Directory not writable: $dir");
}

try {
    ensureDirectory($TEMP_UPLOAD_DIR);
    ensureDirectory($UPLOADS_DIR);
    ensureDirectory($FILES_DIR);
} catch (Exception $e) {
    header('Content-Type: application/json');
    die(json_encode(['success' => false, 'error' => $e->getMessage()]));
}

$endpoint = $_GET['endpoint'] ?? '';

function sanitizeUploadId($id) {
    return preg_replace('/[^a-zA-Z0-9_\.-]/', '', $id);
}

// -----------------------------------------------------------------------------
// ENDPOINT: Init Chunk
// -----------------------------------------------------------------------------
if ($endpoint === 'init_chunk') {
    header('Content-Type: application/json');
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        if (!$input || !isset($input['filename'], $input['filesize'], $input['total_chunks'])) {
            throw new Exception("Invalid request data");
        }
        
        $fileSize = (int)$input['filesize'];
        if ($fileSize > $MAX_UPLOAD_SIZE_BYTES) throw new Exception("File size exceeds limit");
        
        $uploadId = uniqid('chunk_', true);
        $uploadDir = $UPLOADS_DIR . '/' . $uploadId;
        
        if (!mkdir($uploadDir, 0755, true)) throw new Exception("Failed to create upload directory");
        
        $metadata = [
            'upload_id' => $uploadId,
            'filename' => basename($input['filename']),
            'filesize' => $fileSize,
            'total_chunks' => (int)$input['total_chunks'],
            'received_chunks' => 0,
            'created_at' => time()
        ];
        
        file_put_contents($uploadDir . '/metadata.json', json_encode($metadata));
        echo json_encode(['success' => true, 'upload_id' => $uploadId]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Upload Chunk
// -----------------------------------------------------------------------------
if ($endpoint === 'upload_chunk') {
    header('Content-Type: application/json');
    try {
        $uploadId = sanitizeUploadId($_GET['upload_id'] ?? '');
        $chunkIndex = (int)($_GET['chunk_index'] ?? 0);
        
        if (empty($uploadId)) throw new Exception("Missing upload ID");
        
        $uploadDir = $UPLOADS_DIR . '/' . $uploadId;
        if (!is_dir($uploadDir)) throw new Exception("Upload directory not found");
        
        $chunkFile = $uploadDir . '/chunk_' . $chunkIndex;
        $chunkData = file_get_contents('php://input');
        
        if ($chunkData === false) throw new Exception("Failed to read chunk data");
        if (file_put_contents($chunkFile, $chunkData) === false) throw new Exception("Failed to save chunk");
        
        $metadataFile = $uploadDir . '/metadata.json';
        $metadata = json_decode(file_get_contents($metadataFile), true);
        $metadata['received_chunks']++;
        file_put_contents($metadataFile, json_encode($metadata));
        
        if ($metadata['received_chunks'] >= $metadata['total_chunks']) {
            $finalFile = $UPLOADS_DIR . '/' . $uploadId . '_' . basename($metadata['filename']);
            $finalHandle = fopen($finalFile, 'wb');
            
            try {
                for ($i = 0; $i < $metadata['total_chunks']; $i++) {
                    $chunkPath = $uploadDir . '/chunk_' . $i;
                    if (!file_exists($chunkPath)) throw new Exception("Missing chunk $i");
                    $in = fopen($chunkPath, 'rb');
                    while (!feof($in)) fwrite($finalHandle, fread($in, 8192));
                    fclose($in);
                    unlink($chunkPath);
                }
                fclose($finalHandle);
                unlink($metadataFile);
                rmdir($uploadDir);
                chmod($finalFile, 0644);
                
                $queueId = addToQueue($metadata['filename'], $finalFile, $metadata['filesize']);
                echo json_encode(['success' => true, 'upload_id' => $queueId, 'message' => 'Uploaded and queued']);
            } catch (Exception $e) {
                if (is_resource($finalHandle)) fclose($finalHandle);
                if (file_exists($finalFile)) unlink($finalFile);
                throw $e;
            }
        } else {
            echo json_encode(['success' => true, 'message' => 'Chunk received']);
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Standard Upload
// -----------------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_GET['endpoint'])) {
    header('Content-Type: application/json');
    $tempFilePath = null;
    try {
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
            throw new Exception("File upload failed code: " . ($_FILES['file']['error'] ?? 'Unknown'));
        }
        $tempFilePath = $_FILES['file']['tmp_name'];
        if ($_FILES['file']['size'] > $MAX_UPLOAD_SIZE_BYTES) throw new Exception("File too large");
        
        $uploadId = addToQueue($_FILES['file']['name'], $tempFilePath, $_FILES['file']['size']);
        echo json_encode(['success' => true, 'upload_id' => $uploadId, 'message' => 'File queued']);
    } catch (Exception $e) {
        if ($tempFilePath && file_exists($tempFilePath)) @unlink($tempFilePath);
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Status
// -----------------------------------------------------------------------------
if (isset($_GET['endpoint']) && $_GET['endpoint'] === 'status' && isset($_GET['upload_id'])) {
    header('Content-Type: application/json');
    $status = getUploadStatus($_GET['upload_id']);
    if ($status) {
        echo json_encode(['success' => true, 'status' => $status]);
    } else {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Upload not found']);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Process Queue
// -----------------------------------------------------------------------------
if (isset($_GET['endpoint']) && $_GET['endpoint'] === 'process_queue') {
    header('Content-Type: application/json');
    $lockFp = fopen($LOCK_FILE, 'w+');
    if (!flock($lockFp, LOCK_EX | LOCK_NB)) {
        fclose($lockFp);
        echo json_encode(['success' => true, 'worker_busy' => true, 'rate_limit_info' => getRateLimitInfo(), 'counts' => getQueueCounts()]);
        exit;
    }
    try {
        $queue = processQueue();
        echo json_encode([
            'success' => true, 'worker_busy' => false, 'queue' => array_slice($queue, -10),
            'counts' => getQueueCounts(), 'rate_limit_info' => getRateLimitInfo()
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    } finally {
        flock($lockFp, LOCK_UN);
        fclose($lockFp);
    }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: ACK (Delete)
// -----------------------------------------------------------------------------
if (isset($_GET['endpoint']) && $_GET['endpoint'] === 'ack_completion' && isset($_GET['upload_id'])) {
    header('Content-Type: application/json');
    try {
        $uploadId = $_GET['upload_id'];
        updateQueue(function($queue) use ($uploadId) {
            foreach ($queue as $key => $entry) {
                if ($entry['id'] === $uploadId) {
                    if (isset($entry['file_path']) && file_exists($entry['file_path'])) @unlink($entry['file_path']);
                    unset($queue[$key]);
                }
            }
            return array_values($queue);
        });
        echo json_encode(['success' => true]);
    } catch (Exception $e) { echo json_encode(['success' => false, 'error' => $e->getMessage()]); }
    exit;
}

// -----------------------------------------------------------------------------
// ENDPOINT: Purge
// -----------------------------------------------------------------------------
if (isset($_GET['endpoint']) && $_GET['endpoint'] === 'purge_queue') {
    header('Content-Type: application/json');
    try { echo json_encode(purgeQueue()); } 
    catch (Exception $e) { http_response_code(500); echo json_encode(['success' => false, 'error' => $e->getMessage()]); }
    exit;
}

// =============================================================================
// FUNCTIONS
// =============================================================================
function getRateLimitInfo() {
    global $RATE_LIMIT_FILE, $MAX_UPLOADS_PER_MINUTE;
    $fp = fopen($RATE_LIMIT_FILE, 'c+'); flock($fp, LOCK_SH); $content = stream_get_contents($fp); flock($fp, LOCK_UN); fclose($fp);
    $data = $content ? json_decode($content, true) : null; $currentTime = time();
    if (!$data || $currentTime >= $data['reset_time']) return ['uploads_this_minute' => 0, 'reset_time' => $currentTime + 60, 'max_uploads_per_minute' => $MAX_UPLOADS_PER_MINUTE, 'can_upload' => true];
    return ['uploads_this_minute' => $data['uploads_this_minute'], 'reset_time' => $data['reset_time'], 'max_uploads_per_minute' => $MAX_UPLOADS_PER_MINUTE, 'can_upload' => $data['uploads_this_minute'] < $MAX_UPLOADS_PER_MINUTE];
}

function incrementUploadCount() {
    global $RATE_LIMIT_FILE; $fp = fopen($RATE_LIMIT_FILE, 'c+');
    if (flock($fp, LOCK_EX)) {
        $content = stream_get_contents($fp); $data = $content ? json_decode($content, true) : null; $currentTime = time();
        if (!$data || $currentTime >= $data['reset_time']) $data = ['uploads_this_minute' => 1, 'reset_time' => $currentTime + 60];
        else $data['uploads_this_minute']++;
        ftruncate($fp, 0); rewind($fp); fwrite($fp, json_encode($data)); flock($fp, LOCK_UN);
    }
    fclose($fp);
}

function addToQueue($originalFileName, $tempFilePath, $fileSize) {
    global $FILES_DIR; $uploadId = uniqid('vt_', true);
    $permanentPath = $FILES_DIR . '/' . $uploadId . '_' . basename($originalFileName);
    if (!rename($tempFilePath, $permanentPath)) throw new Exception("Failed to move file");
    chmod($permanentPath, 0644);
    $entry = ['id' => $uploadId, 'file_path' => $permanentPath, 'original_name' => $originalFileName, 'file_size' => $fileSize, 'status' => 'queued', 'detailed_status' => 'In Queue', 'created_at' => time(), 'attempts' => 0, 'last_attempt' => null, 'next_attempt' => null];
    updateQueue(function($queue) use ($entry) { $queue[] = $entry; return $queue; });
    return $uploadId;
}

function updateQueue($callback) {
    global $QUEUE_FILE; $fp = fopen($QUEUE_FILE, 'c+');
    if (flock($fp, LOCK_EX)) {
        $content = stream_get_contents($fp); $queue = $content ? json_decode($content, true) : [];
        if (!is_array($queue)) $queue = [];
        $queue = $callback($queue);
        ftruncate($fp, 0); rewind($fp); fwrite($fp, json_encode($queue, JSON_PRETTY_PRINT)); flock($fp, LOCK_UN);
    }
    fclose($fp); return $queue;
}

function loadQueue() {
    global $QUEUE_FILE; if (!file_exists($QUEUE_FILE)) return [];
    $fp = fopen($QUEUE_FILE, 'r'); flock($fp, LOCK_SH); $content = stream_get_contents($fp); flock($fp, LOCK_UN); fclose($fp);
    return json_decode($content, true) ?: [];
}

function getQueueCounts() {
    $q = loadQueue(); $active = 0;
    foreach($q as $e) if(in_array($e['status'], ['queued','retrying','processing'])) $active++;
    return ['active' => $active];
}

function getUploadStatus($uploadId) {
    $queue = loadQueue(); foreach ($queue as $entry) if ($entry['id'] === $uploadId) return $entry;
    return null;
}

function purgeQueue() {
    global $QUEUE_FILE; $deleted = 0;
    updateQueue(function($queue) use (&$deleted) {
        foreach ($queue as $entry) if (isset($entry['file_path']) && file_exists($entry['file_path'])) { @unlink($entry['file_path']); $deleted++; }
        return [];
    });
    return ['message' => "Purged $deleted files.", 'files_deleted' => $deleted];
}

function processQueue() {
    global $VT_API_KEY, $VT_API_URL, $UPLOAD_TIMEOUT, $MAX_RETRIES, $BASE_RETRY_DELAY_SECONDS, 
           $MAX_QUEUE_AGE_HOURS, $MAX_UPLOADS_PER_MINUTE, $FAILED_ENTRY_RETENTION_SECONDS, $VT_DIRECT_LIMIT_BYTES;
    $startTime = time(); $timeLimit = 45; 
    
    updateQueue(function($queue) use (
        $VT_API_KEY, $VT_API_URL, $UPLOAD_TIMEOUT, $MAX_RETRIES, $BASE_RETRY_DELAY_SECONDS, 
        $MAX_QUEUE_AGE_HOURS, $MAX_UPLOADS_PER_MINUTE, $FAILED_ENTRY_RETENTION_SECONDS, $VT_DIRECT_LIMIT_BYTES, $startTime, $timeLimit
    ) {
        $currentTime = time(); $rateInfo = getRateLimitInfo(); $processed = 0;
        $allowed = min($rateInfo['max_uploads_per_minute'] - $rateInfo['uploads_this_minute'], $MAX_UPLOADS_PER_MINUTE);

        foreach ($queue as $key => &$entry) {
            if ((time() - $startTime) > $timeLimit) break;
            if (in_array($entry['status'], ['completed', 'failed'])) {
                if (($currentTime - ($entry['completed_at'] ?? $entry['created_at'])) > $FAILED_ENTRY_RETENTION_SECONDS) {
                    if (isset($entry['file_path']) && file_exists($entry['file_path'])) @unlink($entry['file_path']);
                    unset($queue[$key]); continue;
                }
            }
            if (($currentTime - $entry['created_at']) > ($MAX_QUEUE_AGE_HOURS * 3600)) {
                if (isset($entry['file_path']) && file_exists($entry['file_path'])) @unlink($entry['file_path']);
                unset($queue[$key]); continue;
            }
            if (in_array($entry['status'], ['completed', 'failed'])) continue;
            if ($processed >= $allowed) break;

            if ($entry['status'] === 'queued' || ($entry['status'] === 'retrying' && $currentTime >= $entry['next_attempt'])) {
                $entry['attempts']++; $entry['last_attempt'] = $currentTime; $entry['status'] = 'processing';
                $entry['detailed_status'] = 'Sending to VirusTotal...'; $processed++;
                try {
                    if (!file_exists($entry['file_path'])) throw new Exception("File not found on server");
                    $uploader = new VirusTotalUploader($VT_API_KEY, $VT_API_URL, $UPLOAD_TIMEOUT, $VT_DIRECT_LIMIT_BYTES);
                    $result = $uploader->uploadFile($entry['file_path'], $entry['original_name']);
                    $entry['status'] = 'completed'; $entry['detailed_status'] = 'Upload Complete';
                    $entry['result'] = $result; $entry['completed_at'] = $currentTime; incrementUploadCount();
                    if (file_exists($entry['file_path'])) @unlink($entry['file_path']);
                } catch (Exception $e) {
                    $entry['error'] = $e->getMessage(); $entry['status'] = 'failed'; $entry['detailed_status'] = 'Failed: ' . substr($e->getMessage(), 0, 30) . '...';
                    $transientErrors = ['rate limit', 'cURL error', 'timeout', '429', '502', '503', 'deadline'];
                    foreach ($transientErrors as $err) {
                        if (stripos($e->getMessage(), $err) !== false && $entry['attempts'] < $MAX_RETRIES) {
                            $entry['status'] = 'retrying'; $entry['detailed_status'] = 'Retrying (Error: ' . $err . ')';
                            $entry['next_attempt'] = $currentTime + ($BASE_RETRY_DELAY_SECONDS * pow(2, $entry['attempts'] - 1));
                            break;
                        }
                    }
                    if ($entry['status'] === 'failed') {
                        $entry['completed_at'] = $currentTime;
                        if (file_exists($entry['file_path'])) @unlink($entry['file_path']);
                    }
                }
            }
        }
        return array_values($queue);
    });
    return loadQueue();
}

class VirusTotalUploader {
    private $apiKey; private $apiUrl; private $uploadTimeout; private $limitBytes;
    public function __construct($apiKey, $apiUrl, $uploadTimeout, $limitBytes) {
        $this->apiKey = $apiKey; $this->apiUrl = $apiUrl; $this->uploadTimeout = $uploadTimeout; $this->limitBytes = $limitBytes;
    }
    public function uploadFile($filePath, $originalFileName = null) {
        return (filesize($filePath) > $this->limitBytes) ? $this->largeFileUpload($filePath, $originalFileName) : $this->simpleUpload($filePath, $originalFileName);
    }
    private function simpleUpload($filePath, $originalFileName = null) {
        $curl = curl_init(); $postname = $originalFileName ? $originalFileName : basename($filePath);
        $cFile = new CURLFile($filePath, 'application/octet-stream', $postname);
        curl_setopt_array($curl, [CURLOPT_URL => $this->apiUrl . '/files', CURLOPT_POST => true, CURLOPT_POSTFIELDS => ['file' => $cFile], CURLOPT_HTTPHEADER => ['x-apikey: ' . $this->apiKey, 'Accept: application/json'], CURLOPT_RETURNTRANSFER => true, CURLOPT_TIMEOUT => $this->uploadTimeout]);
        $response = curl_exec($curl); $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE); $error = curl_error($curl); curl_close($curl);
        if ($response === false) throw new Exception("cURL error: " . $error);
        if ($httpCode === 413) return $this->largeFileUpload($filePath, $originalFileName);
        return $this->handleResponse($response, $httpCode);
    }
    private function largeFileUpload($filePath, $originalFileName = null) {
        $curl = curl_init();
        curl_setopt_array($curl, [CURLOPT_URL => $this->apiUrl . '/files/upload_url', CURLOPT_RETURNTRANSFER => true, CURLOPT_HTTPHEADER => ['x-apikey: ' . $this->apiKey, 'Accept: application/json'], CURLOPT_TIMEOUT => 30]);
        $response = curl_exec($curl); $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE); curl_close($curl);
        $data = $this->handleResponse($response, $httpCode, "Get URL"); $uploadUrl = $data['data'];
        $curl = curl_init(); $postname = $originalFileName ? $originalFileName : basename($filePath);
        $cFile = new CURLFile($filePath, 'application/octet-stream', $postname);
        curl_setopt_array($curl, [CURLOPT_URL => $uploadUrl, CURLOPT_POST => true, CURLOPT_POSTFIELDS => ['file' => $cFile], CURLOPT_HTTPHEADER => ['x-apikey: ' . $this->apiKey, 'Accept: application/json'], CURLOPT_RETURNTRANSFER => true, CURLOPT_TIMEOUT => $this->uploadTimeout]);
        $response = curl_exec($curl); $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE); $error = curl_error($curl); curl_close($curl);
        if ($response === false) throw new Exception("Upload failed: " . $error);
        return $this->handleResponse($response, $httpCode, "Upload Phase");
    }
    private function handleResponse($response, $httpCode, $context = "") {
        $contextMsg = $context ? " [$context]" : "";
        if ($httpCode === 401) throw new Exception("Invalid API Key (401)$contextMsg");
        if ($httpCode === 429) throw new Exception("API rate limit exceeded (429)$contextMsg");
        $result = json_decode($response, true);
        if ($httpCode < 200 || $httpCode >= 300) throw new Exception(($result['error']['message'] ?? "HTTP Error $httpCode") . $contextMsg);
        if (!$result) throw new Exception("Invalid JSON response$contextMsg");
        return $result;
    }
}

if (rand(1, 50) === 1) cleanupOldFiles();
function cleanupOldFiles() {
    global $UPLOADS_DIR;
    foreach (glob($UPLOADS_DIR . '/chunk_*', GLOB_ONLYDIR) as $dir) {
        if (time() - filemtime($dir) > 7200) { array_map('unlink', glob("$dir/*")); rmdir($dir); }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>VirusTotal File Uploader</title>
    <link rel="icon" href="data:image/svg+xml,%3Csvg width='1em' height='1em' xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 89'%3E%3Cpath fill='%230b4dda' fill-rule='evenodd' d='M45.292 44.5 0 89h100V0H0l45.292 44.5zM90 80H22l35.987-35.2L22 9h68v71z'/%3E%3C/svg%3E">
    <style>
        :root {
            --bg-color: #f5f5f5; --text-color: #333; --card-bg: #ffffff;
            --card-shadow: 0 2px 10px rgba(0,0,0,0.1); --border-color: #ccc;
            --primary-color: #007cba; --primary-hover: #005a87;
            --accent-bg: #e9ecef; --success-bg: #d4edda; --success-text: #155724;
            --failed-bg: #f8d7da; --failed-text: #721c24; --processing-bg: #fff3cd;
            --drop-hover-bg: #f0f8ff;
        }
        @media (prefers-color-scheme: dark) {
            :root {
                --bg-color: #121212; --text-color: #e0e0e0; --card-bg: #1e1e1e;
                --card-shadow: 0 2px 10px rgba(0,0,0,0.5); --border-color: #444;
                --primary-color: #4da3ff; --primary-hover: #2b80d6;
                --accent-bg: #2d2d2d; --success-bg: #1e4620; --success-text: #a3cfbb;
                --failed-bg: #4c1d21; --failed-text: #ea868f; --processing-bg: #4d420c;
                --drop-hover-bg: #252525;
            }
        }
        body { 
            font-family: Inter, Roboto, "Helvetica Neue", Arial, sans-serif;
            max-width: 1000px; margin: 0 auto; padding: 20px; 
            background-color: var(--bg-color); color: var(--text-color); 
            transition: 0.3s; 
        }
        h1 { margin-top: 0; margin-bottom: 20px; }
        .upload-container { 
            background: var(--card-bg); border-radius: 8px; padding: 30px; 
            box-shadow: var(--card-shadow); text-align: center; 
        }
        .drop-area { 
            border: 3px dashed var(--border-color); border-radius: 8px; 
            padding: 40px; cursor: pointer; transition: 0.3s; margin-bottom: 20px; 
        }
        .drop-area:hover, .drop-area.dragover, .drop-area:focus-visible { 
            border-color: var(--primary-color); background-color: var(--drop-hover-bg); 
            outline: none; 
        }
        .upload-btn { 
            background-color: var(--primary-color); color: white; border: none; 
            padding: 12px 24px; border-radius: 4px; cursor: pointer; margin: 10px 0; 
            pointer-events: none; /* Let clicks pass to the container */
        }
        .info-container { display: flex; gap: 20px; margin: 20px 0; }
        .info-column { 
            flex: 1; padding: 10px; background-color: var(--accent-bg); 
            border-radius: 4px; text-align: center; 
        }
        .info-column h3 { margin: 0 0 5px 0; font-size: 14px; }
        .info-column p { margin: 2px 0; font-size: 13px; }
        
        .file-list { text-align: left; margin-top: 20px; }
        .file-item { 
            padding: 10px; border: 1px solid var(--border-color); 
            border-radius: 4px; margin-bottom: 8px; background-color: var(--card-bg); 
        }
        .file-name { font-family: monospace; font-weight: bold; margin-bottom: 4px; display: block; }
        .file-status { font-size: 14px; }
        .file-item.completed { background-color: var(--success-bg); color: var(--success-text); }
        .file-item.failed { background-color: var(--failed-bg); color: var(--failed-text); }
        .file-item.processing { background-color: var(--processing-bg); }
        
        .purge-btn { 
            background-color: #dc3545; color: white; border: none; 
            padding: 8px 16px; border-radius: 4px; cursor: pointer; width: 100%; 
            font-size: 13px;
        }
        .purge-btn:hover { background-color: #bb2d3b; }
        *:focus-visible { outline: 2px solid var(--primary-color); outline-offset: 2px; }
        a { color: var(--primary-color); }
    </style>
</head>
<body>
    <main class="upload-container">
        <header>
            <h1>VirusTotal File Uploader</h1>
        </header>
        
        <div class="drop-area" id="dropArea" tabindex="0" role="button" aria-label="Drop files here or click to upload">
            <p>Drag & drop files here</p>
            <div class="upload-btn">Browse Files</div>
            <input type="file" id="fileInput" style="display:none" multiple>
        </div>
        
        <section class="info-container" aria-label="System Status">
            <div class="info-column">
                <h3>Status</h3>
                <p id="queueStatusText" aria-live="polite">Idle</p>
                <p>Active: <span id="totalCount">0</span></p>
            </div>
            <div class="info-column">
                <h3>Rate Limit</h3>
                <p>Used: <span id="uploadsCount">0</span>/<span id="maxUploads">4</span></p>
                <p>Reset: <span id="resetTime">-</span></p>
            </div>
            <div class="info-column">
                <h3>Action</h3>
                <button class="purge-btn" id="purgeBtn">Purge Queue</button>
            </div>
        </section>
        
        <section class="file-list" id="fileListContainer" aria-live="polite" aria-relevant="additions">
            <!-- File items added here -->
        </section>
    </main>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const dropArea = document.getElementById('dropArea');
            const fileInput = document.getElementById('fileInput');
            const fileListContainer = document.getElementById('fileListContainer');
            
            const MAX_CONCURRENT_UPLOADS = 2;
            let pendingQueue = []; 
            let activeUploads = 0; 

            // Click handling for drop area
            dropArea.addEventListener('click', () => fileInput.click());
            
            // Keyboard access for drop area
            dropArea.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    fileInput.click();
                }
            });

            fileInput.addEventListener('change', handleFileSelect);
            document.getElementById('purgeBtn').addEventListener('click', purgeQueue);

            // Drag & Drop visuals
            ['dragenter', 'dragover'].forEach(evt => {
                dropArea.addEventListener(evt, (e) => { 
                    e.preventDefault(); 
                    dropArea.classList.add('dragover'); 
                });
            });
            ['dragleave', 'drop'].forEach(evt => {
                dropArea.addEventListener(evt, (e) => { 
                    e.preventDefault(); 
                    dropArea.classList.remove('dragover'); 
                });
            });

            dropArea.addEventListener('drop', (e) => {
                if(e.dataTransfer.files.length) handleFiles(e.dataTransfer.files);
            });

            function handleFileSelect(e) { if (e.target.files.length) handleFiles(e.target.files); }
            
            function handleFiles(files) {
                Array.from(files).forEach(file => {
                    const id = 'ui-' + Math.random().toString(36).substr(2, 9);
                    addToFileList(id, file.name);
                    queueUpload(file, id);
                });
            }

            function queueUpload(file, uiId) {
                pendingQueue.push({file: file, id: uiId});
                updateStatus(uiId, "Waiting in queue...");
                processQueue();
            }

            function processQueue() {
                if (activeUploads >= MAX_CONCURRENT_UPLOADS) return;
                if (pendingQueue.length === 0) return;

                const task = pendingQueue.shift();
                activeUploads++;
                
                performUpload(task.file, task.id)
                    .finally(() => {
                        activeUploads--;
                        processQueue();
                    });
            }

            function performUpload(file, uiId) {
                return new Promise((resolve) => {
                    const onComplete = () => resolve();
                    const browserLimit = <?php echo $BROWSER_CHUNK_SIZE_BYTES; ?>;
                    
                    if (file.size > browserLimit) {
                        uploadChunked(file, uiId, onComplete);
                    } else {
                        uploadStandard(file, uiId, onComplete);
                    }
                });
            }

            function addToFileList(id, name) {
                const el = document.createElement('div');
                el.className = 'file-item'; el.id = id;
                el.innerHTML = `<span class="file-name">${name}</span><span class="file-status">Preparing...</span>`;
                fileListContainer.prepend(el);
            }

            function updateStatus(id, msg, type = '') {
                const el = document.getElementById(id);
                if (el) {
                    el.querySelector('.file-status').innerHTML = msg;
                    if(type) el.className = `file-item ${type}`;
                }
            }

            function uploadStandard(file, uiId, callback) {
                const formData = new FormData();
                formData.append('file', file);
                const xhr = new XMLHttpRequest();
                xhr.upload.addEventListener('progress', (e) => {
                    if(e.lengthComputable) updateStatus(uiId, `Uploading to Server: ${Math.round((e.loaded/e.total)*100)}%`);
                });
                xhr.onload = () => {
                    try {
                        const r = JSON.parse(xhr.responseText);
                        if(r.success) { updateStatus(uiId, 'Queued for VirusTotal...'); pollStatus(r.upload_id, uiId); }
                        else updateStatus(uiId, 'Error: '+r.error, 'failed');
                    } catch(e) { updateStatus(uiId, 'Server Error', 'failed'); }
                    callback();
                };
                xhr.onerror = () => { updateStatus(uiId, 'Network Error', 'failed'); callback(); };
                xhr.open('POST', '');
                xhr.send(formData);
            }

            function uploadChunked(file, uiId, callback) {
                const chunkSize = <?php echo $BROWSER_CHUNK_SIZE_BYTES; ?>;
                const totalChunks = Math.ceil(file.size / chunkSize);
                
                fetch('?endpoint=init_chunk', {
                    method: 'POST',
                    body: JSON.stringify({filename: file.name, filesize: file.size, total_chunks: totalChunks})
                }).then(r=>r.json()).then(d => {
                    if(!d.success) throw new Error(d.error);
                    sendChunk(d.upload_id, 0);
                }).catch(e => {
                    updateStatus(uiId, 'Init Fail: '+e.message, 'failed');
                    callback();
                });

                function sendChunk(uid, idx) {
                    const start = idx * chunkSize;
                    const chunk = file.slice(start, Math.min(start + chunkSize, file.size));
                    
                    fetch(`?endpoint=upload_chunk&upload_id=${uid}&chunk_index=${idx}`, {
                        method: 'POST', body: chunk
                    }).then(r=>r.json()).then(d => {
                        if(!d.success) throw new Error(d.error);
                        const percent = Math.round(((idx + 1) / totalChunks) * 100);
                        updateStatus(uiId, `Uploading to Server: ${percent}%`);
                        
                        if(d.upload_id && d.message.includes('queued')) {
                            updateStatus(uiId, 'Queued for VirusTotal...');
                            pollStatus(d.upload_id, uiId);
                            callback();
                        } else {
                            sendChunk(uid, idx+1);
                        }
                    }).catch(e => {
                        updateStatus(uiId, 'Chunk Fail: '+e.message, 'failed');
                        callback();
                    });
                }
            }

            function pollStatus(sid, uid) {
                const interval = setInterval(() => {
                    fetch(`?endpoint=status&upload_id=${sid}`).then(r=>r.json()).then(d => {
                        if(d.success && d.status) {
                            const s = d.status;
                            if(s.status === 'completed') {
                                clearInterval(interval);
                                const aid = s.result.data.id;
                                updateStatus(uid, `Done. <a href="https://www.virustotal.com/gui/file-analysis/${aid}" target="_blank">View Analysis</a>`, 'completed');
                                fetch(`?endpoint=ack_completion&upload_id=${sid}`);
                            } else if(s.status === 'failed') {
                                clearInterval(interval);
                                updateStatus(uid, 'Failed: '+s.error, 'failed');
                                fetch(`?endpoint=ack_completion&upload_id=${sid}`);
                            } else {
                                const msg = s.detailed_status || `Processing (Attempt ${s.attempts})`;
                                updateStatus(uid, msg, 'processing');
                            }
                        } else { clearInterval(interval); }
                    }).catch(() => clearInterval(interval));
                }, 4000);
            }

            function purgeQueue() {
                if(confirm('Clear queue history? Active uploads will stop.')) {
                    fetch('?endpoint=purge_queue').then(r=>r.json()).then(d => {
                        fileListContainer.innerHTML = '';
                        pendingQueue = []; 
                        activeUploads = 0;
                        alert(d.message);
                    });
                }
            }

            setInterval(() => {
                fetch('?endpoint=process_queue').then(r=>r.json()).then(d => {
                    if(d.success) {
                        document.getElementById('totalCount').innerText = d.counts ? d.counts.active : 0;
                        document.getElementById('uploadsCount').innerText = d.rate_limit_info.uploads_this_minute;
                        
                        const statusText = d.worker_busy ? 'Processing Queue...' : 
                                         (d.counts && d.counts.active > 0) ? 'Waiting for worker...' : 'Idle';
                        document.getElementById('queueStatusText').innerText = statusText;
                        
                        if(d.rate_limit_info.reset_time) {
                            document.getElementById('resetTime').innerText = new Date(d.rate_limit_info.reset_time * 1000).toLocaleTimeString();
                        }
                    }
                });
            }, 5000);
        });
    </script>
</body>
</html>
